<?php
declare(strict_types=1);

/**
 * Vision Media Bot — router.php (final, no handle_start)
 * فقط توابع کمکی روت/کیبورد و ابزارهای مشترک
 */

// ——— Inline Keyboard Helpers ———
function kb_btn(string $text, string $data): array { return ['text'=>$text,'callback_data'=>$data]; }
function kb_url(string $text, string $url): array { return ['text'=>$text,'url'=>$url]; }
function kb_row(array $buttons): array { return $buttons; }
function kb_inline(array $rows): array { return ['inline_keyboard'=>$rows]; }

// ——— Role-based Panel Router ———
function route_to_panel(int $chat_id, array $user): void {
    $role = $user['role'] ?? null;
    if (!$role) {
        $rows = [
            [kb_btn(icon('employee')." کارمند هستم", "emp:req")],
            [kb_btn(icon('customer')." مشتری هستم", "cust:req")],
        ];
        tg_send_kb_html(
            $chat_id,
            vm_polite_header($user['full_name']) .
            "شما هنوز هیچ نقشی در سیستم ندارید 🙃\n"
            . "لطفاً نقش‌تان را انتخاب کنید:",
            kb_inline($rows)
        );
        return;
    }
    switch ($role) {
        case 'CEO':            ceo_menu($chat_id, $user); break;
        case 'ACCOUNTING':     acc_menu($chat_id, $user['id']); break;
        case 'DEPT_MANAGER':   dept_menu($chat_id, $user); break;
        case 'ADMIN_LEAD':
        case 'ADMIN':
        case 'VIDEOGRAPHER':
        case 'BLOGGER':
        case 'EDITOR':
        case 'SCENARIO_WRITER': employee_menu($chat_id, $user); break;
        case 'CUSTOMER':       customer_menu($chat_id, $user); break;
        default: tg_send_html($chat_id, "❌ نقش ناشناخته: {$role}");
    }
}

// ——— Command Parser for Callbacks ———
function parse_command(array $parts): array {
    return [
        'root' => $parts[0] ?? '',
        'a'    => $parts[1] ?? '',
        'b'    => $parts[2] ?? '',
        'c'    => $parts[3] ?? '',
    ];
}

// ——— Penalty Auto Helper ———
function penalties_auto(int $uid, string $reason): void {
    $user = user_get($uid);
    if (!$user) return;

    $pen = [
        'id'     => uniqid('pen_'),
        'uid'    => $uid,
        'name'   => $user['full_name'] ?? '',
        'reason' => $reason,
        'amount' => PENALTY_PUBLIC_DEFAULT,
        'time'   => time(),
        'auto'   => true,
    ];
    storage_append(STORAGE['penalties'], $pen);

    $msg = str_replace(
        ['{date}','{name}','{reason}','{amount}'],
        [jdate('Y/m/d H:i'), $pen['name'], $pen['reason'], number_format($pen['amount'])],
        MESSAGES['penalty_format']
    );

    tg_send_html($uid, $msg);
    tg_send_html(CEO_ID, "🔔 جریمه خودکار ثبت شد برای {$pen['name']}");
    $accs = users_by_role('ACCOUNTING');
    foreach ($accs as $a) tg_send_html((int)$a['id'], "🔔 جریمه خودکار ثبت شد برای {$pen['name']}");
}

// ——— Task Helpers ———
function tasks_user(int $uid): array {
    $all = storage_load(STORAGE['tasks']);
    return array_values(array_filter($all, fn($t) => (int)$t['uid'] === $uid));
}
function tasks_flag_warn(int $uid, string $tid, string $flag): void {
    $all = storage_load(STORAGE['tasks']);
    foreach ($all as &$t) if ($t['uid']==$uid && $t['id']===$tid) $t[$flag]=true;
    storage_save(STORAGE['tasks'], $all);
}
function tasks_mark_done(int $uid, string $tid): void {
    $all = storage_load(STORAGE['tasks']);
    foreach ($all as &$t) if ($t['uid']==$uid && $t['id']===$tid) $t['done']=time();
    storage_save(STORAGE['tasks'], $all);
}

// ——— Customer Payment Helpers ———
function customer_payment_add(array $cust, int $amount, string $status='prepay', string $due=''): void {
    $all = storage_load(STORAGE['invoices']);
    $all[] = [
        'id'=>uniqid('inv_'),'cust'=>$cust['id']??0,'name'=>$cust['full_name']??'',
        'amount'=>$amount,'status'=>$status,'due'=>$due,'time'=>time(),
    ];
    storage_save(STORAGE['invoices'], $all);
}
function customer_invoices(int $cust_id): array {
    $all = storage_load(STORAGE['invoices']);
    return array_values(array_filter($all, fn($i)=>(int)$i['cust']===$cust_id));
}
function customer_invoice_remind(array $inv, array $cust): void {
    $msg = str_replace(
        ['{remain}','{date}'],
        [number_format($inv['amount']??0), $inv['due'] ?: jdate('Y/m/d')],
        PAYMENT_MSG['remind_due']
    );
    tg_send_html((int)$cust['id'], $msg);
}

// ——— Utility: Jalali date fallback ———
if (!function_exists('jdate')) {
    function jdate(string $format, ?int $ts=null): string {
        $ts = $ts ?? time();
        return date($format, $ts);
    }
}

// --- پایان router.php ---
