<?php
declare(strict_types=1);

/**
 * Vision Media Bot — handlers/customer.php (FULL, minimal)
 * پنل مشتری: سرویس‌ها، فروشگاه، فاکتور‌ها + دیسپچر مکالمه برای ثبت رسید
 */

/* ======================= UI: منوی مشتری ======================= */

function customer_menu(int $chat_id, array $user): void {
    $rows = [
        [ kb_btn("💼 سرویس‌های من", "cust:services"), kb_btn("🛍️ فروشگاه", "cust:store") ],
        [ kb_btn("💳 فاکتورها", "cust:invoices"),     kb_btn("🔄 تمدید سرویس", "cust:services") ],
    ];
    tg_send_kb_html(
        $chat_id,
        vm_polite_header($user['full_name']) . "🤝 <b>پنل مشتری</b>\nیکی از گزینه‌ها را انتخاب کنید:",
        kb_inline($rows)
    );
}

/* ======================= Router اصلی مشتری ======================= */

function customer_router(int $chat_id, int $uid, array $user, array $parts): void {
    // درخواست نقش (وقتی کاربر تازه استارت می‌زند)
    if (($parts[1] ?? '') === 'req') {
        $pendings = approvals_by_target($uid);
        foreach ($pendings as $p) {
            if (($p['type'] ?? '') === 'role_req' && ($p['status'] ?? '') === 'pending') {
                tg_send_info($chat_id, "درخواست قبلی‌ات نزد حسابداری در انتظار بررسی است 🌱");
                return;
            }
        }
        $id = approval_create('role_req', $uid, ['want' => 'customer']);
        tg_send_success($chat_id, "درخواستت برای «مشتری» ثبت شد. حسابداری بررسی می‌کند 🌹");
        $accs = users_by_role('ACCOUNTING');
        foreach ($accs as $a) {
            tg_send_html((int)$a['id'], "📥 درخواست نقش جدید (مشتری)\nID کاربر: {$uid}\nApproval: {$id}\nاز «درخواست‌های نقش» اقدام کنید.");
        }
        return;
    }

    $act = $parts[1] ?? '';

    if ($act === 'services') { customer_services($chat_id, $uid); return; }
    if ($act === 'store')    { customer_store_menu($chat_id);   return; }
    if ($act === 'sview')    { $sid = (int)($parts[2] ?? 0); customer_store_view($chat_id, $uid, $sid); return; }
    if ($act === 'invoices') { customer_invoices_list($chat_id, $uid); return; }

    // پیش‌فرض: منو
    customer_menu($chat_id, $user);
}

/* ======================= سرویس‌های مشتری ======================= */

function customer_services(int $chat_id, int $uid): void {
    // در این نسخه‌ی مینیمال، فقط جای‌نما:
    $svcs = storage_load(STORAGE['services_active']);
    $mine = array_values(array_filter($svcs, fn($s) => (int)($s['cust'] ?? 0) === $uid));

    if (!$mine) {
        tg_send_html($chat_id, "فعلاً هیچ سرویس فعالی برای شما ثبت نشده.\nبرای خرید یا تمدید از «🛍️ فروشگاه» استفاده کن.");
        return;
    }

    $msg = "💼 <b>سرویس‌های فعال شما</b>\n\n";
    foreach ($mine as $r) {
        $msg .= "• " . htmlspecialchars((string)($r['title'] ?? 'سرویس')) . "\n";
        if (!empty($r['expires'])) $msg .= "  └ تاریخ پایان: " . htmlspecialchars((string)$r['expires']) . "\n";
    }
    tg_send_html($chat_id, $msg);
}

/* ======================= فروشگاه برای مشتری ======================= */

function customer_store_menu(int $chat_id): void {
    if (empty(SERVICES)) { tg_send_html($chat_id, "فعلاً محصولی در فروشگاه ثبت نشده."); return; }

    $msg = "🛍️ <b>فروشگاه خدمات ویژن مدیا</b>\nبرای مشاهده‌ی جزئیات هر خدمت، روی دکمه‌اش بزنید:";
    $rows = [];
    foreach (SERVICES as $s) {
        $rows[] = [ kb_btn("«{$s['title']}» — " . money((int)$s['price']), "cust:sview:" . $s['id']) ];
    }
    tg_send_kb_html($chat_id, $msg, kb_inline($rows));
}

function customer_store_view(int $chat_id, int $uid, int $sid): void {
    $svc = arr_find(SERVICES, fn($x) => (int)$x['id'] === $sid);
    if (!$svc) { tg_send_alert($chat_id, "سرویس پیدا نشد."); return; }

    $price = money((int)$svc['price']);
    $msg = "🛍️ <b>{$svc['title']}</b>\n\n"
         . ($svc['desc'] ?? '') . "\n\n"
         . "💵 قیمت: <b>{$price}</b>\n\n"
         . "برای خرید کارت‌به‌کارت:\n"
         . "🔴 کارت: <code>" . PAYTO['card'] . "</code>\n"
         . "🔵 حساب: <code>" . PAYTO['account'] . "</code>\n"
         . "🟣 شبا: <code>" . PAYTO['iban'] . "</code>\n"
         . "⚪ به نام: " . PAYTO['holder'] . "\n\n"
         . "بعد از پرداخت، روی دکمه‌ی «ارسال رسید پرداخت» بزن و عکس/متن رسید رو بفرست ❤️";

    $rows = [
        [ kb_btn("📤 ارسال رسید پرداخت", "cust:sendrcpt:" . $sid) ],
        [ kb_btn("↩️ بازگشت به فروشگاه", "cust:store") ],
    ];
    tg_send_kb_html($chat_id, $msg, kb_inline($rows));

    // فعال‌سازی فلو ارسال رسید پرداخت این سرویس
    conv_start($uid, 'cust_send_receipt', 0, ['sid' => $sid]);
}

/* ======================= فاکتورهای مشتری ======================= */

function customer_invoices_list(int $chat_id, int $uid): void {
    $invs = customer_invoices($uid);
    if (!$invs) { tg_send_html($chat_id, "هیچ فاکتوری برای شما ثبت نشده."); return; }

    $msg = "💳 <b>فاکتورهای شما</b>\n\n";
    foreach ($invs as $inv) {
        $msg .= "• " . htmlspecialchars((string)$inv['id']) . " — مبلغ: " . money((int)$inv['amount']) . "\n";
        $msg .= "  وضعیت: " . htmlspecialchars((string)$inv['status']) . " | تاریخ: " . dt_jalali((int)$inv['time']) . "\n";
        if (!empty($inv['due'])) $msg .= "  سررسید: " . htmlspecialchars((string)$inv['due']) . "\n";
        $msg .= "\n";
    }
    tg_send_html($chat_id, $msg);
}

/* ======================= دیسپچر مکالمه‌ی مشتری ======================= */

function customer_conv_dispatch_text(int $chat_id, int $uid, string $text): bool {
    $s = conv_get($uid); if (!$s) return false;
    $flow = $s['flow'] ?? '';

    // فلو ارسال رسید پرداخت (بعد از مشاهده جزئیات سرویس)
    if ($flow === 'cust_send_receipt') {
        // اینجا می‌تونیم عکس/متن رسید را بگیریم؛
        // چون این فایل فقط متن را می‌گیرد، به‌صورت مینیمال ثبت می‌کنیم:
        $sid = (int)($s['data']['sid'] ?? 0);
        $note = trim($text);
        $payload = [
            'cust' => $uid,
            'sid'  => $sid,
            'note' => $note,
            'time' => time(),
        ];
        // ذخیره‌سازی رسید به شکل ساده
        $receipts = storage_load(STORAGE['receipts']);
        $receipts[] = $payload;
        storage_save(STORAGE['receipts'], $receipts);

        // اطلاع به حسابداری
        $accs = users_by_role('ACCOUNTING');
        foreach ($accs as $a) {
            tg_send_html((int)$a['id'], "🧾 رسید پرداخت جدید از مشتری {$uid}\nسرویس: {$sid}\nتوضیح/شماره پیگیری: {$note}");
        }

        conv_clear($uid);
        tg_send_success($chat_id, "✅ رسید پرداختت ثبت شد. تیم حسابداری بررسی و تایید می‌کند.");
        return true;
    }

    return false; // متن مربوط به فلو مشتری نبود
}
