<?php
declare(strict_types=1);

/**
 * VisionMedia Bot — core/telegram.php (FINAL)
 * - tg_request: لایه‌ی واحد برای فراخوانی متدهای تلگرام (curl + fallback)
 * - توابع کمکی: ارسال/ویرایش/حذف پیام، اکشن تایپینگ، عکس/سند، وبهوک
 * - لاگ خطاها در storage/logs/telegram.log
 */

// ────────── پیکربندی توکن ──────────
// انتظار می‌رود در config/env.php تعریف شده باشد.
// از هر دو نام پشتیبانی می‌کنیم:
if (!defined('BOT_TOKEN') && defined('TELEGRAM_BOT_TOKEN')) {
    define('BOT_TOKEN', TELEGRAM_BOT_TOKEN);
}
if (!defined('BOT_TOKEN')) {
    // اگر هنوز هم تعریف نشده بود، اجرای ربات منطقی نیست.
    // ولی برای جلوگیری از Fatal، یک مقدار خالی می‌گذاریم و لاگ می‌کنیم.
    if (!function_exists('app_log')) {
        function app_log(string $msg): void {
            $f = __DIR__ . '/../storage/logs/app.log';
            if (!is_dir(dirname($f))) { @mkdir(dirname($f), 0775, true); }
            @file_put_contents($f, '['.date('Y-m-d H:i:s')."] ".$msg."\n", FILE_APPEND);
        }
    }
    app_log('BOT_TOKEN is not defined! Please set in config/env.php');
    define('BOT_TOKEN', '');
}

// ────────── ابزار لاگ محلی تلگرام ──────────
if (!function_exists('tg_log')) {
    function tg_log(string $msg): void {
        $f = __DIR__ . '/../storage/logs/telegram.log';
        if (!is_dir(dirname($f))) { @mkdir(dirname($f), 0775, true); }
        @file_put_contents($f, '['.date('Y-m-d H:i:s')."] ".$msg."\n", FILE_APPEND);
    }
}

// ────────── آدرس پایهٔ API ──────────
if (!function_exists('tg_api_url')) {
    function tg_api_url(string $method): string {
        return 'https://api.telegram.org/bot'. BOT_TOKEN .'/'. $method;
    }
}

/**
 * tg_request
 * متد عمومی برای فراخوانی API تلگرام
 * - اگر curl باشد از curl استفاده می‌کند؛ در غیر این صورت از file_get_contents با stream context
 * - $params آرایهٔ معمولی است؛ برای فایل‌ها از ['field'=>'@/path/to/file'] استفاده نکن، از tg_send_photo/tg_send_document کمک بگیر.
 * - خروجی: آرایهٔ decode شدهٔ JSON (یا آرایهٔ ['ok'=>false,'error'=>'...'] در خطا)
 */
if (!function_exists('tg_request')) {
    function tg_request(string $method, array $params = []): array {
        if (BOT_TOKEN === '') {
            tg_log("NO BOT_TOKEN for method {$method}");
            return ['ok'=>false,'error'=>'no-bot-token'];
        }

        $url = tg_api_url($method);

        // تشخیص ارسال multipart (وقتی فایل داریم) — اگر فایل resource یا CURLFile باشد.
        $multipart = false;
        foreach ($params as $k=>$v) {
            if ($v instanceof CURLFile) { $multipart = true; break; }
            // اگر مسیر فایل لوکال ست شده باشد و وجود داشته باشد، در توابع کمکی تبدیلش می‌کنیم.
        }

        // اگر curl موجود است:
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);

            if ($multipart) {
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
            } else {
                // ارسال معمولی (application/x-www-form-urlencoded)
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
                curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded; charset=UTF-8']);
            }

            $resp = curl_exec($ch);
            if ($resp === false) {
                $err = curl_error($ch);
                curl_close($ch);
                tg_log("curl_error({$method}): ".$err);
                return ['ok'=>false,'error'=>$err];
            }
            $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            $json = json_decode((string)$resp, true);
            if (!is_array($json)) {
                tg_log("invalid_json({$method}) code={$code} resp=".$resp);
                return ['ok'=>false,'error'=>'invalid-json','http_code'=>$code];
            }
            if (!($json['ok'] ?? false)) {
                tg_log("tg_error({$method}) code={$code} ".json_encode($json, JSON_UNESCAPED_UNICODE));
            }
            return $json;
        }

        // fallback: file_get_contents
        $opts = [
            'http' => [
                'method'  => 'POST',
                'header'  => "Content-Type: application/x-www-form-urlencoded; charset=UTF-8\r\n",
                'content' => http_build_query($params),
                'timeout' => 30
            ]
        ];
        $context = stream_context_create($opts);
        $resp = @file_get_contents($url, false, $context);
        if ($resp === false) {
            tg_log("fopen_error({$method})");
            return ['ok'=>false,'error'=>'http-open-failed'];
        }
        $json = json_decode((string)$resp, true);
        if (!is_array($json)) {
            tg_log("invalid_json_fopen({$method}) resp=".$resp);
            return ['ok'=>false,'error'=>'invalid-json'];
        }
        if (!($json['ok'] ?? false)) {
            tg_log("tg_error_fopen({$method}) ".json_encode($json, JSON_UNESCAPED_UNICODE));
        }
        return $json;
    }
}
/* ادامه: core/telegram.php — توابع کمکی ارسال/ویرایش/فایل/وبهوک */

// ────────── اکشن تایپینگ ──────────
if (!function_exists('tg_typing')) {
    function tg_typing(int $chat_id, string $action='typing'): void {
        tg_request('sendChatAction', [
            'chat_id' => $chat_id,
            'action'  => $action
        ]);
    }
}

// ────────── ویرایش و حذف پیام ──────────
if (!function_exists('tg_edit_html')) {
    function tg_edit_html(int $chat_id, int $message_id, string $text, ?array $kb=null): array {
        $p = [
            'chat_id' => $chat_id,
            'message_id' => $message_id,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];
        if ($kb) $p['reply_markup'] = json_encode($kb, JSON_UNESCAPED_UNICODE);
        return tg_request('editMessageText', $p);
    }
}
if (!function_exists('tg_delete')) {
    function tg_delete(int $chat_id, int $message_id): array {
        return tg_request('deleteMessage', [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ]);
    }
}

// ────────── ارسال عکس/سند (با پشتیبانی از فایل لوکال و URL) ──────────
if (!function_exists('tg_send_photo')) {
    function tg_send_photo(int $chat_id, string $file_or_url, string $caption='', ?array $kb=null): array {
        $params = [
            'chat_id' => $chat_id,
            'caption' => $caption,
            'parse_mode' => 'HTML'
        ];
        if (preg_match('~^https?://~i', $file_or_url)) {
            // URL
            $params['photo'] = $file_or_url;
        } else {
            // فایل لوکال
            if (is_file($file_or_url)) {
                $params['photo'] = new CURLFile($file_or_url);
            } else {
                // به عنوان URL نامعتبر ارسال نکنیم؛ یک ارور لاگ می‌کنیم:
                tg_log("tg_send_photo: file not found: ".$file_or_url);
                $params['photo'] = $file_or_url; // تلگرام خطا می‌دهد؛ حداقل درخواست ثبت می‌شود.
            }
        }
        if ($kb) $params['reply_markup'] = json_encode($kb, JSON_UNESCAPED_UNICODE);
        return tg_request('sendPhoto', $params);
    }
}
if (!function_exists('tg_send_document')) {
    function tg_send_document(int $chat_id, string $file_or_url, string $caption='', ?array $kb=null): array {
        $params = [
            'chat_id' => $chat_id,
            'caption' => $caption,
            'parse_mode' => 'HTML'
        ];
        if (preg_match('~^https?://~i', $file_or_url)) {
            $params['document'] = $file_or_url;
        } else {
            if (is_file($file_or_url)) {
                $params['document'] = new CURLFile($file_or_url);
            } else {
                tg_log("tg_send_document: file not found: ".$file_or_url);
                $params['document'] = $file_or_url;
            }
        }
        if ($kb) $params['reply_markup'] = json_encode($kb, JSON_UNESCAPED_UNICODE);
        return tg_request('sendDocument', $params);
    }
}

// ────────── وبهوک (اختیاری) ──────────
if (!function_exists('tg_set_webhook')) {
    /**
     * tg_set_webhook
     * $url: آدرس کامل اسکریپت وبهوک (HTTPS)
     * $allowed_updates: آرایه‌ای از تایپ‌های مجاز (message, edited_message, callback_query, my_chat_member, chat_member, ...)
     */
    function tg_set_webhook(string $url, array $allowed_updates = ['message','edited_message','callback_query','my_chat_member','chat_member']): array {
        return tg_request('setWebhook', [
            'url' => $url,
            'allowed_updates' => json_encode($allowed_updates, JSON_UNESCAPED_UNICODE)
        ]);
    }
}

if (!function_exists('tg_delete_webhook')) {
    function tg_delete_webhook(): array {
        return tg_request('deleteWebhook', []);
    }
}

// ────────── ارسال ساده HTML (اگر utils هنوز لود نشده باشد) ──────────
if (!function_exists('tg_send_html')) {
    function tg_send_html(int $chat_id, string $text): void {
        tg_request('sendMessage', [
            'chat_id' => $chat_id,
            'text' => $text,
            'parse_mode' => 'HTML'
        ]);
    }
}
if (!function_exists('tg_send_kb_html')) {
    function tg_send_kb_html(int $chat_id, string $text, array $kb): void {
        tg_request('sendMessage', [
            'chat_id' => $chat_id,
            'text' => $text,
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode($kb, JSON_UNESCAPED_UNICODE)
        ]);
    }
}
