<?php
declare(strict_types=1);
/**
 * VisionMedia Bot - index.php (v17)
 * - No vm_get_* function declarations here (avoids redeclare).
 * - Reads config via config_boot if present; otherwise direct from storage/config.json.
 * - Global error/shutdown handler prints readable text instead of HTTP 500.
 * - Optional modules loaded via @include_once; router checks function_exists before calling.
 */

error_reporting(E_ALL);
ini_set('display_errors','0');

function vm17_error($msg){
    header('Content-Type: text/plain; charset=utf-8');
    http_response_code(200);
    echo "VM-BOT ERROR: ".$msg;
    exit;
}
set_error_handler(function($no,$str,$file,$line){ vm17_error("PHP[$no] $str @ $file:$line"); });
register_shutdown_function(function(){
    $e = error_get_last();
    if ($e && in_array($e['type'], [E_ERROR,E_PARSE,E_CORE_ERROR,E_COMPILE_ERROR])) {
        vm17_error("FATAL {$e['message']} @ {$e['file']}:{$e['line']}");
    }
});

$BASE_DIR = __DIR__;
@include_once $BASE_DIR . '/config.php';
@include_once $BASE_DIR . '/functions/utils.php';
@include_once $BASE_DIR . '/functions/settings.php';
@include_once $BASE_DIR . '/functions/config_boot.php'; // optional

// Resolve config WITHOUT declaring vm_get_* here
$VM_CFG = [];
$STORAGE_DIR = $BASE_DIR . '/storage';
if (function_exists('vm_config_load')) {
    $VM_CFG = vm_config_load();
} else {
    if (!is_dir($STORAGE_DIR)) @mkdir($STORAGE_DIR, 0775, true);
    $raw = @file_get_contents($STORAGE_DIR . '/config.json');
    $VM_CFG = @json_decode($raw ?: "{}", true) ?: [];
}
$VM_TOKEN = isset($VM_CFG['BOT_TOKEN']) ? (string)$VM_CFG['BOT_TOKEN'] : (defined('BOT_TOKEN') ? BOT_TOKEN : '');
$VM_BASE  = isset($VM_CFG['BASE_URL']) ? (string)$VM_CFG['BASE_URL'] : (defined('BASE_URL') ? BASE_URL : '');
$VM_TZ    = isset($VM_CFG['TIMEZONE']) ? (string)$VM_CFG['TIMEZONE'] : (defined('TIMEZONE') ? TIMEZONE : 'Asia/Tehran');

// Optional includes (NEVER fatal)
@include_once $BASE_DIR . '/functions/telegram.php';
@include_once $BASE_DIR . '/functions/storage.php';
@include_once $BASE_DIR . '/functions/users.php';
@include_once $BASE_DIR . '/functions/approvals.php';
@include_once $BASE_DIR . '/functions/role_requests.php';
@include_once $BASE_DIR . '/functions/penalties.php';
@include_once $BASE_DIR . '/functions/photographer_upload_review.php';
@include_once $BASE_DIR . '/functions/offish.php';
@include_once $BASE_DIR . '/functions/tasks.php';
@include_once $BASE_DIR . '/functions/store.php';
@include_once $BASE_DIR . '/functions/penalties_flow.php';
@include_once $BASE_DIR . '/functions/payroll.php';
@include_once $BASE_DIR . '/handlers/start.php';
@include_once $BASE_DIR . '/handlers/role.php';
@include_once $BASE_DIR . '/handlers/ceo.php';
@include_once $BASE_DIR . '/handlers/accounting.php';
@include_once $BASE_DIR . '/handlers/department.php';
@include_once $BASE_DIR . '/handlers/employee.php';
@include_once $BASE_DIR . '/handlers/customer.php';
@include_once $BASE_DIR . '/handlers/dm.php';

@date_default_timezone_set($VM_TZ);
if (function_exists('storage_boot')) { storage_boot(); }

// Minimal tg_api if not provided
if (!function_exists('tg_api')) {
    function tg_api(string $method, array $params=[]): array {
        global $VM_TOKEN;
        if (!$VM_TOKEN) return ['ok'=>false,'error'=>'token not set'];
        $url = 'https://api.telegram.org/bot'.$VM_TOKEN.'/'.$method;
        $ctx = stream_context_create(['http'=>['method'=>'POST','header'=>"Content-Type: application/json\r\n",'content'=>json_encode($params)]]);
        $res = @file_get_contents($url,false,$ctx);
        $j = @json_decode($res ?: "{}", true);
        return is_array($j) ? $j : ['ok'=>false,'error'=>'no json'];
    }
}
if (!function_exists('tg_send_html')) {
    function tg_send_html(int $chat_id, string $text, array $kb=null): array {
        $rp = $kb ? json_encode(['inline_keyboard'=>$kb], JSON_UNESCAPED_UNICODE) : null;
        return tg_api('sendMessage', ['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>'HTML','disable_web_page_preview'=>true,'reply_markup'=>$rp]);
    }
}

// Helpers
function out_json($a){ header('Content-Type: application/json; charset=utf-8'); echo json_encode($a,JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT); exit; }
function ok(){ echo 'ok'; exit; }

// Setup (no vm_get_*; direct)
if (isset($_GET['setup']) && $_GET['setup']=='1') {
    if (!is_dir($STORAGE_DIR)) @mkdir($STORAGE_DIR, 0775, true);
    $curT = htmlspecialchars($VM_TOKEN ?? '', ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8');
    $curB = htmlspecialchars($VM_BASE ?? '', ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8');
    $curZ = htmlspecialchars($VM_TZ ?? 'Asia/Tehran', ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8');
    if ($_SERVER['REQUEST_METHOD']==='POST') {
        $t = trim($_POST['token'] ?? '');
        $b = trim($_POST['base'] ?? '');
        $z = trim($_POST['tz'] ?? 'Asia/Tehran');
        if ($t && $b) {
            @file_put_contents($STORAGE_DIR.'/config.json', json_encode(['BOT_TOKEN'=>$t,'BASE_URL'=>$b,'TIMEZONE'=>$z], JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
            echo "<meta charset='utf-8'><p>ذخیره شد ✅ — حالا <a href='?setWebhook=1'>setWebhook</a> را بزن.</p>"; exit;
        } else {
            echo "<meta charset='utf-8'><p style='color:red'>هر دو مقدار لازم است.</p>";
        }
    }
    echo "<meta charset='utf-8'><style>body{font-family:sans-serif;max-width:680px;margin:40px auto}</style>";
    echo "<h2>VisionMedia Bot — Setup</h2>";
    echo "<form method='post'>";
    echo "<label>BOT_TOKEN</label><br><input name='token' value='{$curT}' style='width:100%'><br><br>";
    echo "<label>BASE_URL (تا index.php)</label><br><input name='base' value='{$curB}' style='width:100%'><br><br>";
    echo "<label>TIMEZONE</label><br><input name='tz' value='{$curZ}' style='width:100%'><br><br>";
    echo "<button type='submit'>ذخیره</button></form>";
    exit;
}

// Diagnostics
if (isset($_GET['ping'])) { echo 'pong'; exit; }
if (isset($_GET['version'])) { echo 'VisionMedia v17'; exit; }
if (isset($_GET['debug'])) {
  header('Content-Type: text/plain; charset=utf-8');
  echo "UPDATE RAW:\n".(file_get_contents('php://input')?:'');
  exit;
}
if (isset($_GET['selftest'])) {
    $checks = [
        'php'=>PHP_VERSION,
        'token_set'=>(bool)$VM_TOKEN,
        'base_set'=>(bool)$VM_BASE,
        'storage_writable'=>is_writable($STORAGE_DIR ?: __DIR__),
        'time'=>date('c'),
    ];
    out_json(['ok'=>true,'checks'=>$checks]);
}
if (isset($_GET['health'])) { out_json(['ok'=>true,'time'=>date('c')]); }

// Webhook
if (isset($_GET['setWebhook'])) { if (!$VM_BASE) vm17_error("BASE_URL empty (run ?setup=1)"); out_json(tg_api('setWebhook', ['url'=>$VM_BASE])); }
if (isset($_GET['delWebhook'])) { out_json(tg_api('deleteWebhook', [])); }
if (isset($_GET['seed'])) {
    if (function_exists('users_all') && function_exists('users_save')) {
        $u = users_all();
        if (!isset($u[(string)CEO_ID])) $u[(string)CEO_ID] = ['id'=>CEO_ID,'first'=>'CEO','role'=>'ceo','created_at'=>time()];
        users_save($u); echo "seeded"; exit;
    }
    echo "seed skipped"; exit;
}

// Token guard
if (!$VM_TOKEN) vm17_error("Token not set. Use ?setup=1 to configure.");

// Read update
$raw = file_get_contents('php://input') ?: '';
$update = json_decode($raw, true);
if (!is_array($update)) $update = [];

// Callback router
if (isset($update['callback_query'])) {
    $cb = $update['callback_query'];
    $chat_id = (int)($cb['message']['chat']['id'] ?? ($cb['from']['id'] ?? 0));
    $data = (string)($cb['data'] ?? '');
    if (!$chat_id || !$data) ok();
    try {
        if ($data==='back:home' && function_exists('handle_cmd_start')) handle_cmd_start(['message'=>['chat'=>['id'=>$chat_id],'from'=>$cb['from']??[]]]);
        elseif (strpos($data,'ceo:')===0 && function_exists('ceo_handle')) ceo_handle($data,$chat_id);
        elseif (strpos($data,'acc:')===0) {
            if (function_exists('acc_handle_extras') && acc_handle_extras($data,$chat_id)) { /* ok */ }
            elseif (function_exists('acc_handle_invoice_cb') && acc_handle_invoice_cb($data,$chat_id)) { /* ok */ }
            elseif (function_exists('acc_store_list') && ($data==='acc:store' || $data==='acc:store:list')) acc_store_list($chat_id);
            elseif (function_exists('acc_store_item') && strpos($data,'acc:store:item:')===0) acc_store_item($chat_id,(int)substr($data,strlen('acc:store:item:')));
            elseif (function_exists('acc_store_add') && $data==='acc:store:add') acc_store_add($chat_id);
            elseif (function_exists('acc_store_toggle') && strpos($data,'acc:store:toggle:')===0) acc_store_toggle($chat_id,(int)substr($data,strlen('acc:store:toggle:')));
            elseif (function_exists('acc_store_begin_edit') && strpos($data,'acc:store:edit:')===0) { $p=explode(':',$data); acc_store_begin_edit($chat_id,(int)$p[4],(string)$p[3]); }
            elseif (function_exists('acc_handle')) acc_handle($data,$chat_id);
        }
        elseif (strpos($data,'dm:')===0) {
            if (function_exists('dm_handle_calendar_root') && dm_handle_calendar_root($data,$chat_id)) { /* ok */ }
            elseif (function_exists('dm_handle_reports') && dm_handle_reports($data,$chat_id)) { /* ok */ }
            elseif (function_exists('dm_handle_plus') && dm_handle_plus($data,$chat_id)) { /* ok */ }
            elseif (function_exists('dm_handle')) dm_handle($data,$chat_id);
        }
        elseif (strpos($data,'emp:')===0 && function_exists('emp_handle')) emp_handle($data,$chat_id);
        elseif (strpos($data,'cust:')===0 && function_exists('cust_handle')) cust_handle($data,$chat_id);
        elseif (strpos($data,'offish:arrive:')===0 && function_exists('offish_mark_arrival')) { $p=explode(':',$data); $ans=$p[2]??'no'; $oid=(int)($p[3]??0); $role=$p[4]??''; offish_mark_arrival($oid,$role,$ans==='yes'); }
        elseif (strpos($data,'pf:')===0 && function_exists('pf_handle_callback')) pf_handle_callback($chat_id,$data);
        else { tg_send_html($chat_id, "دستور ناشناخته."); }
    } catch (Throwable $e) { tg_send_html($chat_id, "⛔️ خطای داخلی (callback) روی: <code>".htmlspecialchars($data ?? "", ENT_QUOTES|ENT_SUBSTITUTE,"UTF-8")."</code>"); }
    ok();
}

// Message router
if (isset($update['message'])) {
    $msg = $update['message']; $chat_id = (int)($msg['chat']['id'] ?? 0); $text = trim((string)($msg['text'] ?? ''));
    try {
        if (function_exists('acc_store_handle_edit_message') && acc_store_handle_edit_message($chat_id,$msg)) ok();
        if (function_exists('pf_handle_message') && pf_handle_message($chat_id,$msg)) ok();
        if ($text==='/start' && function_exists('handle_cmd_start')) handle_cmd_start(['message'=>$msg]);
        elseif ($text==='/dm' && function_exists('dm_menu')) dm_menu($chat_id);
        elseif ($text==='/acc' && function_exists('acc_menu')) acc_menu($chat_id);
        elseif ($text==='/ceo' && function_exists('ceo_menu')) ceo_menu($chat_id);
        elseif ($text==='/health') tg_send_html($chat_id, "<b>Health OK</b>");
        else { if (function_exists('handle_cmd_start')) handle_cmd_start(['message'=>$msg]); else tg_send_html($chat_id, "سلام! /start را بزن."); }
    } catch (Throwable $e) { tg_send_html($chat_id, "⛔️ خطای داخلی (message)."); }
    ok();
}

ok();
