<?php
declare(strict_types=1);

/* --- Compatibility shim for old CEO handler --- */
if (!defined('ACC_INSTALLED')) define('ACC_INSTALLED', true);
if (!function_exists('accounting_is_installed')) {
    function accounting_is_installed(): bool { return true; }
}
if (!function_exists('open_accounting_panel')) {
    // بعضی نسخه‌های قدیمی ceo از این اسم صدا می‌زنند
    function open_accounting_panel(int $chat_id): void {
        if (function_exists('accountant_menu_main')) {
            accountant_menu_main($chat_id);
        } else {
            tg_send_html($chat_id, '⚠️ ماژول حسابداری لود نشده.');
        }
    }
}

/**
 * VisionMedia — CEO Panel (FINAL)
 * امکانات:
 * - 👥 پرسنل (لیست/پروفایل/تغییر نقش/فعال‌سازی/جریمه/پاداش)
 * - 🤝 مشتریان (لیست/پروفایل/اتصال مدیر دپارتمان/فاکتورهای مشتری)
 * - 🏢 دپارتمان‌ها (تخصیص مدیر برای مشتری)
 * - 🧾 حسابداری (پنل کامل حسابداری؛ مدیرعامل = دسترسی کامل)
 * - 📊 داشبورد و 📝 گزارش‌های سریع
 * - 📣 اعلامیه سراسری (Broadcast)
 * - 🗂 ممیزی/لاگ
 * - ⚙️ تنظیمات (پشتیبان‌گیری/خروجی JSON، ساعات کاری، SLA نوتیف)
 */

require_once __DIR__ . '/../core/utils.php';
require_once __DIR__ . '/../core/telegram.php';
require_once __DIR__ . '/../core/storage.php';
require_once __DIR__ . '/../config/env.php';

require_once __DIR__ . '/../modules/accounting.php';  // inv_*
require_once __DIR__ . '/../modules/department.php';  // dept_*
require_once __DIR__ . '/../modules/penalties.php';   // pen_*

// اگر پروژه str_starts ندارد، اینجا تضمین می‌کنیم:
if (!function_exists('str_starts')) {
    function str_starts(string $haystack, string $needle): bool {
        return strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

/* ────────── ثوابت ────────── */
const CEO_PAGE_SIZE       = 10;
const CEO_STATE_FILE      = __DIR__ . '/../storage/ceo_state.json';
const BROADCAST_FILE      = __DIR__ . '/../storage/broadcasts.json';
const CEO_SETTINGS_FILE   = __DIR__ . '/../storage/ceo_settings.json';
const EXPORT_DIR          = __DIR__ . '/../storage/exports';

/* ────────── دسترسی ────────── */
function ceo_is_ceo(int $user_id): bool { return $user_id === CEO_UID; }

/* ────────── State ────────── */
function ceo_state_load(): array {
    if (!is_file(CEO_STATE_FILE)) return [];
    $j = json_decode((string)@file_get_contents(CEO_STATE_FILE), true);
    return is_array($j) ? $j : [];
}
function ceo_state_save(array $s): bool { return (bool)@file_put_contents(CEO_STATE_FILE, json_encode($s, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT)); }
function ceo_state_set(int $uid, string $key, $val): void { $s=ceo_state_load(); $s[$uid][$key]=$val; ceo_state_save($s); }
function ceo_state_get(int $uid, string $key, $def=null){ $s=ceo_state_load(); return $s[$uid][$key] ?? $def; }
function ceo_state_clear(int $uid): void { $s=ceo_state_load(); unset($s[$uid]); ceo_state_save($s); }

/* ────────── Settings ────────── */
function ceo_settings_load(): array {
    if (!is_file(CEO_SETTINGS_FILE)) return [
        'sla_alerts' => true,
        'business_hours' => '09-18',
    ];
    $j = json_decode((string)@file_get_contents(CEO_SETTINGS_FILE), true);
    return is_array($j) ? $j : ['sla_alerts'=>true,'business_hours'=>'09-18'];
}
function ceo_settings_save(array $cfg): bool {
    return (bool)@file_put_contents(CEO_SETTINGS_FILE, json_encode($cfg, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
}

/* ────────── UI Helpers ────────── */
function ceo_paginate_kb(string $base, int $page, int $pages): array {
    if ($pages<=1) return kb_inline([]);
    $row=[];
    if ($page>1) $row[] = kb_btn('◀️ قبلی', "{$base}:p:".($page-1));
    if ($page<$pages) $row[] = kb_btn('بعدی ▶️', "{$base}:p:".($page+1));
    return kb_inline([ $row ]);
}
function ceo_back_kb(string $to): array { return kb_inline([[ kb_btn('◀️ بازگشت', $to) ]]); }

/* ────────── منوی اصلی ────────── */
if (!function_exists('ceo_menu_main')) {
    function ceo_menu_main(int $chat_id): void {
        $kb = kb_inline([
            [ kb_btn('👥 پرسنل', 'ceo:staff'), kb_btn('🤝 مشتریان', 'ceo:customers') ],
            [ kb_btn('🏢 دپارتمان‌ها', 'ceo:departments'), kb_btn('🧾 حسابداری', 'ceo:accounting') ],
            [ kb_btn('📊 داشبورد', 'ceo:dashboard'), kb_btn('📝 گزارش‌ها', 'ceo:reports') ],
            [ kb_btn('📣 اعلامیه سراسری', 'ceo:broadcast'), kb_btn('🗂 لاگ و ممیزی', 'ceo:audit') ],
            [ kb_btn('⚙️ تنظیمات', 'ceo:settings') ],
        ]);
        $msg = "👑 <b>خوش اومدی مدیرعامل عزیز!</b>\n"
             . "مرکز فرماندهی ویژن‌مدیا ✨\n"
             . "هر بخشی که می‌خوای مدیریت کنی رو انتخاب کن:";
        tg_send_kb_html($chat_id, $msg, $kb);
    }
}

/* ────────── 👥 پرسنل ────────── */
function ceo_staff_list(int $chat_id, int $page=1): void {
    $all = storage_load(EMPLOYEES_FILE, []);
    if (!$all) { tg_send_kb_html($chat_id,'هنوز پرسنلی ثبت نشده.', ceo_back_kb('ceo:back')); return; }
    usort($all, fn($a,$b)=>strcmp(($a['full_name']??''), ($b['full_name']??'')));
    $total=count($all); $pages=(int)ceil($total/CEO_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($all, ($page-1)*CEO_PAGE_SIZE, CEO_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $e) {
        $uid=(int)($e['chat_id']??0);
        $name = $e['full_name'] ?: ('کارمند '.to_persian_num($uid));
        $job  = $e['job'] ?: '—';
        $rows[] = [ kb_btn("👤 {$name} — {$job}", "ceo:staff:show:{$uid}") ];
    }
    $nav = ceo_paginate_kb('ceo:staff:list', $page, $pages)['inline_keyboard'] ?? [];
    if ($nav) $rows = array_merge($rows, $nav);
    $rows[] = [ kb_btn('➕ افزودن پرسنل (راهنما)', 'ceo:staff:add') ];
    $rows[] = [ kb_btn('◀️ بازگشت', 'ceo:back') ];

    tg_send_kb_html($chat_id, "👥 <b>لیست پرسنل</b> (صفحه ".to_persian_num($page)."/".to_persian_num($pages).")", ['inline_keyboard'=>$rows]);
}
function ceo_staff_profile(int $chat_id, int $uid): void {
    $emps = storage_load(EMPLOYEES_FILE, []);
    $e=null; foreach ($emps as $x) if ((int)($x['chat_id']??0)===$uid) { $e=$x; break; }
    if (!$e) { tg_send_html($chat_id,'⛔️ کارمند یافت نشد.'); return; }

    $name = tg_escape($e['full_name'] ?: '—');
    $job  = tg_escape($e['job'] ?: '—');
    $active = ($e['active'] ?? true) ? 'فعال 🟢' : 'غیرفعال 🔴';

    $ded=0; foreach (($e['deductions']??[]) as $d) $ded+=(int)($d['amount']??0);
    $bon=0; foreach (($e['bonuses']??[])    as $b) $bon+=(int)($b['amount']??0);

    $msg = "👤 <b>{$name}</b>\n"
         . "نقش: <b>{$job}</b>\n"
         . "آی‌دی: <code>".to_persian_num($uid)."</code>\n"
         . "وضعیت: {$active}\n"
         . "کسرها: <b>".money($ded)."</b> | پاداش‌ها: <b>".money($bon)."</b>";

    $kb = kb_inline([
        [ kb_btn('✏️ تغییر نقش', "ceo:staff:role:{$uid}"), kb_btn(($e['active']??true)?'🔒 غیرفعال‌سازی':'🔓 فعال‌سازی', "ceo:staff:toggle:{$uid}") ],
        [ kb_btn('⚠️ جریمه سریع', "ceo:staff:pen:{$uid}"), kb_btn('🎁 پاداش سریع', "ceo:staff:rew:{$uid}") ],
        [ kb_btn('◀️ بازگشت به پرسنل', 'ceo:staff') ],
    ]);
    tg_send_kb_html($chat_id, $msg, $kb);
}
function ceo_staff_toggle(int $uid): bool {
    $emps = storage_load(EMPLOYEES_FILE, []);
    foreach ($emps as &$e) if ((int)($e['chat_id']??0)===$uid) { $e['active']=!($e['active']??true); return storage_save(EMPLOYEES_FILE,$emps); }
    return false;
}
function ceo_staff_set_role(int $uid, string $role): bool {
    $emps = storage_load(EMPLOYEES_FILE, []);
    foreach ($emps as &$e) if ((int)($e['chat_id']??0)===$uid) { $e['job']=$role; return storage_save(EMPLOYEES_FILE,$emps); }
    return false;
}
function ceo_staff_pen_quick(int $chat_id, int $uid, int $amount, string $reason): void {
    $pid = pen_submit_manual($uid,$amount,$reason,CEO_UID,null,null,'ceo');
    tg_send_html($chat_id, "✅ جریمه #".to_persian_num($pid)." برای کاربر ".to_persian_num($uid)." ثبت شد و برای تاییدها ارسال گردید.");
}
function ceo_staff_rew_quick(int $uid, int $amount, string $reason): bool {
    $emps = storage_load(EMPLOYEES_FILE, []);
    foreach ($emps as &$e) if ((int)($e['chat_id']??0)===$uid) {
        $b=$e['bonuses']??[]; $b[]=['amount'=>$amount,'reason'=>$reason,'ts'=>time()]; $e['bonuses']=$b;
        return storage_save(EMPLOYEES_FILE,$emps);
    }
    return false;
}

/* ────────── 🤝 مشتریان ────────── */
function ceo_customers_list(int $chat_id, int $page=1): void {
    $all = storage_load(CUSTOMERS_FILE, []);
    if (!$all) { tg_send_kb_html($chat_id,'هنوز مشتری ثبت نشده.', ceo_back_kb('ceo:back')); return; }
    usort($all, fn($a,$b)=>strcmp(($a['brand']??''), ($b['brand']??'')));
    $total=count($all); $pages=(int)ceil($total/CEO_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($all, ($page-1)*CEO_PAGE_SIZE, CEO_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $c) {
        $cid=(int)($c['chat_id']??0);
        $brand=$c['brand'] ?: 'بدون‌نام'; $ig=$c['instagram'] ?: '—';
        $rows[] = [ kb_btn("🏢 {$brand} — {$ig}", "ceo:cust:show:{$cid}") ];
    }
    $nav=ceo_paginate_kb('ceo:cust:list',$page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('➕ افزودن مشتری (راهنما)', 'ceo:cust:add') ];
    $rows[]=[ kb_btn('◀️ بازگشت', 'ceo:back') ];

    tg_send_kb_html($chat_id, "🤝 <b>لیست مشتریان</b> (صفحه ".to_persian_num($page)."/".to_persian_num($pages).")", ['inline_keyboard'=>$rows]);
}
function ceo_customer_profile(int $chat_id, int $cust_id): void {
    $all = storage_load(CUSTOMERS_FILE, []);
    $c=null; foreach ($all as $x) if ((int)($x['chat_id']??0)===$cust_id) { $c=$x; break; }
    if (!$c) { tg_send_html($chat_id,'⛔️ مشتری یافت نشد.'); return; }

    $brand = tg_escape($c['brand'] ?: '—');
    $ig = tg_escape($c['instagram'] ?: '—');
    $team = $c['team'] ?? []; $mgr=(int)($team['manager']??0);

    $msg = "🏢 <b>{$brand}</b>\n"
         . "اینستاگرام: <code>{$ig}</code>\n"
         . "آی‌دی مشتری: <code>".to_persian_num($cust_id)."</code>\n"
         . "مدیر دپارتمان: <b>".($mgr? to_persian_num($mgr):'—')."</b>";

    $kb = kb_inline([
        [ kb_btn('👑 انتخاب/تغییر مدیر دپارتمان', "ceo:dept:assign:start:{$cust_id}") ],
        [ kb_btn('🧾 فاکتورهای مشتری', "ceo:acc:inv:for:{$cust_id}") ],
        [ kb_btn('◀️ بازگشت به مشتریان', 'ceo:customers') ],
    ]);
    tg_send_kb_html($chat_id,$msg,$kb);
}

/* ────────── 🏢 دپارتمان‌ها ────────── */
function ceo_dept_pick_manager_kb(int $cust_id, int $page=1): array {
    $emps = storage_load(EMPLOYEES_FILE, []);
    $mgrs = array_values(array_filter($emps, function($e){
        $j=mb_strtolower((string)($e['job']??'')); return str_contains($j,'مدیر') || str_contains($j,'manager');
    }));
    if (!$mgrs) return ceo_back_kb($cust_id? "ceo:cust:show:{$cust_id}" : 'ceo:departments');

    usort($mgrs, fn($a,$b)=>strcmp(($a['full_name']??''), ($b['full_name']??'')));
    $total=count($mgrs); $pages=(int)ceil($total/CEO_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($mgrs, ($page-1)*CEO_PAGE_SIZE, CEO_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $m) {
        $uid=(int)($m['chat_id']??0);
        $name=$m['full_name'] ?: '—';
        $rows[]=[ kb_btn("👤 {$name} (".to_persian_num($uid).")", "ceo:dept:assign:set:{$cust_id}:{$uid}") ];
    }
    $nav=ceo_paginate_kb("ceo:dept:assign:pick:{$cust_id}", $page, $pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows, $nav);
    $rows[]=[ kb_btn('◀️ بازگشت', $cust_id? "ceo:cust:show:{$cust_id}" : 'ceo:departments') ];

    return ['inline_keyboard'=>$rows];
}

/* ────────── 🧾 حسابداری برای مدیرعامل ────────── */

/* --- Compatibility shim for old CEO handler --- */
if (!defined('ACC_INSTALLED')) define('ACC_INSTALLED', true);
if (!function_exists('accounting_is_installed')) {
    function accounting_is_installed(): bool { return true; }
}
if (!function_exists('open_accounting_panel')) {
    // بعضی نسخه‌های قدیمی ceo از این اسم صدا می‌زنند
    function open_accounting_panel(int $chat_id): void {
        if (function_exists('accountant_menu_main')) {
            accountant_menu_main($chat_id);
        } else {
            tg_send_html($chat_id, '⚠️ ماژول حسابداری لود نشده.');
        }
    }
}

function ceo_acc_list_invoices(int $chat_id, int $page=1, ?int $cust_id=null): void {
    $invoices = inv_list();
    if ($cust_id) $invoices = array_values(array_filter($invoices, fn($i)=>(int)($i['customer_chat_id']??0)===$cust_id));
    if (!$invoices) {
        tg_send_kb_html($chat_id, $cust_id? 'هیچ فاکتوری برای این مشتری نیست.' : 'هنوز فاکتوری ثبت نشده.', ceo_back_kb($cust_id? "ceo:cust:show:{$cust_id}" : 'ceo:accounting'));
        return;
    }
    usort($invoices, fn($a,$b)=>($b['created_ts']??0) <=> ($a['created_ts']??0));

    $paid_cnt=0; $open_cnt=0; $remain_total=0;
    foreach ($invoices as $inv) {
        $st=$inv['status']??'unpaid';
        if ($st==='paid') $paid_cnt++; else $open_cnt++;
        $info=inv_remain_info((int)$inv['id']); $remain_total+=(int)($info['remain']??0);
    }

    $total=count($invoices); $pages=(int)ceil($total/CEO_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($invoices, ($page-1)*CEO_PAGE_SIZE, CEO_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $inv) {
        $id=(int)$inv['id']; $st=$inv['status']??'unpaid'; $remain=inv_remain_info($id);
        $rows[]=[ kb_btn("🧾 #".to_persian_num($id)." — {$st} — مانده: ".money((int)$remain['remain']), "ceo:acc:inv:view:{$id}") ];
    }
    $nav=ceo_paginate_kb($cust_id? "ceo:acc:inv:for:{$cust_id}" : 'ceo:acc:inv', $page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);

    $rows[]=[ kb_btn('🧾 پنل فاکتورها (حسابداری)', 'acc:invoices') ];
    $rows[]=[ kb_btn('💼 دفترکل', 'acc:cashbook'), kb_btn('📝 صورتجلسه', 'acc:minutes') ];
    $rows[]=[ kb_btn('📊 گزارش‌ها', 'acc:reports') ];
    $rows[]=[ kb_btn('◀️ بازگشت', $cust_id? "ceo:cust:show:{$cust_id}" : 'ceo:accounting') ];

    $title = ($cust_id? "🧾 فاکتورهای مشتری ".to_persian_num($cust_id) : "🧾 فهرست فاکتورها")
           . "\n— پرداخت‌شده: <b>".to_persian_num($paid_cnt)."</b> | باز/ناتمام: <b>".to_persian_num($open_cnt)."</b>"
           . "\n— مانده کل باز: <b>".money($remain_total)."</b>";

    tg_send_kb_html($chat_id, $title, ['inline_keyboard'=>$rows]);
}
function ceo_acc_view_invoice(int $chat_id, int $inv_id): void {
    $inv = inv_find($inv_id);
    if (!$inv) { tg_send_html($chat_id,'⛔️ فاکتور یافت نشد.'); return; }
    $view = inv_render_view_for_customer($inv_id);
    $kb = kb_inline([
        [ kb_btn('✔️ تسویه سریع', "ceo:acc:inv:settle:{$inv_id}") ],
        [ kb_btn('📄 باز کردن در پنل حسابداری', "acc:inv:view:{$inv_id}") ],
        [ kb_btn('◀️ بازگشت', 'ceo:acc:invoices') ],
    ]);
    tg_send_kb_html($chat_id, $view, $kb);
}
function ceo_acc_settle_quick(int $chat_id, int $inv_id): void {
    $r = inv_remain_info($inv_id);
    $ok = inv_apply_payment($inv_id, (int)$r['remain']);
    tg_send_html($chat_id, $ok? "✅ فاکتور #".to_persian_num($inv_id)." تسویه شد." : "⛔️ تسویه نشد.");
    ceo_acc_view_invoice($chat_id,$inv_id);
}

/* ────────── 📊 داشبورد و 📝 گزارش ────────── */
function ceo_dashboard_render(int $chat_id): void {
    $emps = storage_load(EMPLOYEES_FILE, []);
    $cust = storage_load(CUSTOMERS_FILE, []);
    $invs = inv_list();
    $staff_active = count(array_filter($emps, fn($e)=> ($e['active'] ?? true)));
    $customers_active = count($cust);
    $open_invs = array_filter($invs, fn($i)=> in_array(($i['status']??'unpaid'), ['unpaid','partial'], true));
    $open_count = count($open_invs);
    $remain_total = 0;
    foreach ($open_invs as $inv) { $info = inv_remain_info((int)$inv['id']); $remain_total += (int)$info['remain']; }
    $msg = "📊 <b>داشبورد لحظه‌ای</b>\n"
         . "👥 پرسنل فعال: <b>".to_persian_num($staff_active)."</b>\n"
         . "🤝 مشتریان: <b>".to_persian_num($customers_active)."</b>\n"
         . "🧾 فاکتورهای باز: <b>".to_persian_num($open_count)."</b>\n"
         . "💰 مانده کل: <b>".money($remain_total)."</b>";
    tg_send_kb_html($chat_id, $msg, ceo_back_kb('ceo:back'));
}
function ceo_report_quick(int $chat_id, string $scope='روزانه'): void {
    $invs = inv_list();
    $open = count(array_filter($invs, fn($i)=> in_array(($i['status']??'unpaid'), ['unpaid','partial'], true)));
    $paid = count(array_filter($invs, fn($i)=> ($i['status']??'')==='paid'));
    $msg = "📝 <b>گزارش {$scope}</b>\n"
         . "🧾 پرداخت‌شده: <b>".to_persian_num($paid)."</b>\n"
         . "🧾 باز/ناتمام: <b>".to_persian_num($open)."</b>";
    tg_send_kb_html($chat_id,$msg, ceo_back_kb('ceo:reports'));
}

/* ────────── 📣 Broadcast ────────── */
function ceo_broadcast_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('👥 همهٔ کارمندان', 'ceo:bc:aud:employees'), kb_btn('👑 مدیران دپارتمان', 'ceo:bc:aud:managers') ],
        [ kb_btn('🤝 همهٔ مشتریان', 'ceo:bc:aud:customers') ],
        [ kb_btn('◀️ بازگشت', 'ceo:back') ]
    ]);
    tg_send_kb_html($chat_id, "📣 <b>اعلامیه سراسری</b>\nمخاطب را انتخاب کن:", $kb);
}
function ceo_broadcast_collect_targets(string $audience): array {
    if ($audience==='employees') {
        $emps = storage_load(EMPLOYEES_FILE, []); return array_values(array_map(fn($e)=>(int)($e['chat_id']??0), $emps));
    }
    if ($audience==='managers') {
        $emps = storage_load(EMPLOYEES_FILE, []);
        $mgrs = array_filter($emps, function($e){ $j=mb_strtolower((string)($e['job']??'')); return str_contains($j,'مدیر')||str_contains($j,'manager'); });
        return array_values(array_map(fn($e)=>(int)($e['chat_id']??0), $mgrs));
    }
    if ($audience==='customers') {
        $cust = storage_load(CUSTOMERS_FILE, []); return array_values(array_map(fn($c)=>(int)($c['chat_id']??0), $cust));
    }
    return [];
}
function ceo_broadcast_start(int $chat_id, int $uid, string $aud): void {
    $targets = ceo_broadcast_collect_targets($aud);
    if (!$targets) { tg_send_kb_html($chat_id, "⛔️ مخاطبی یافت نشد.", ceo_back_kb('ceo:broadcast')); return; }
    ceo_state_set($uid, 'bc', ['aud'=>$aud,'targets'=>$targets,'step'=>'await_text']);
    tg_send_kb_html($chat_id, "لطفاً متن اعلامیه را ارسال کن ✍️\n(ارسال که شد، برای ".to_persian_num(count($targets))." نفر می‌رود.)", ceo_back_kb('ceo:broadcast'));
}
function ceo_broadcast_send(int $chat_id, int $uid, string $text): void {
    $st = ceo_state_get($uid,'bc'); if (!$st || ($st['step']??'')!=='await_text') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $targets = $st['targets'] ?? [];
    $ok=0; foreach ($targets as $tid) { $r = tg_send_html((int)$tid, "📣 <b>پیام مدیرعامل:</b>\n".$text); if (($r['ok']??false)===true) $ok++; }
    $bcs = is_file(BROADCAST_FILE) ? (json_decode((string)@file_get_contents(BROADCAST_FILE), true) ?: []) : [];
    $bcs[] = ['by'=>$uid,'audience'=>$st['aud'],'count'=>count($targets),'sent_ok'=>$ok,'ts'=>time(),'text'=>$text];
    @file_put_contents(BROADCAST_FILE, json_encode($bcs, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
    ceo_state_clear($uid);
    tg_send_kb_html($chat_id, "✅ ارسال شد. موفق: <b>".to_persian_num($ok)."</b> / ".to_persian_num(count($targets)), ceo_back_kb('ceo:broadcast'));
}

/* ────────── 🗂 Audit ────────── */
function ceo_audit_show(int $chat_id, int $lines=50): void {
    $log = __DIR__ . '/../storage/logs/app.log';
    if (!is_file($log)) { tg_send_kb_html($chat_id,'هنوز لاگی ثبت نشده.', ceo_back_kb('ceo:back')); return; }
    $content = trim((string)@file_get_contents($log));
    $arr = preg_split('~\R~', $content); $arr = array_slice($arr, -$lines);
    $text = "<b>آخرین ".to_persian_num(count($arr))." رویداد</b>:\n<code>".tg_escape(implode("\n",$arr))."</code>";
    tg_send_kb_html($chat_id, $text, ceo_back_kb('ceo:back'));
}

/* ────────── ⚙️ تنظیمات ────────── */
function ceo_settings_menu(int $chat_id): void {
    $cfg = ceo_settings_load();
    $sla = $cfg['sla_alerts'] ? 'روشن 🟢' : 'خاموش 🔴';
    $hrs = $cfg['business_hours'] ?? '09-18';
    $kb = kb_inline([
        [ kb_btn("🔔 هشدار SLA: {$sla}", 'ceo:settings:sla:toggle') ],
        [ kb_btn("🕰 ساعات کاری: {$hrs}", 'ceo:settings:hours') ],
        [ kb_btn('💾 خروجی/پشتیبان‌گیری', 'ceo:settings:export') ],
        [ kb_btn('◀️ بازگشت', 'ceo:back') ],
    ]);
    $msg = "⚙️ <b>تنظیمات مدیرعامل</b>\n"
         . "— هشدار نزدیک‌بودن به موعد وظایف (SLA)\n"
         . "— تعیین ساعات کاری سازمان (برای برنامه‌ریزی وظایف)\n"
         . "— دریافت خروجی JSON از داده‌های کلیدی";
    tg_send_kb_html($chat_id, $msg, $kb);
}
function ceo_settings_hours_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('09-18', 'ceo:settings:hours:set:09-18'), kb_btn('08-17', 'ceo:settings:hours:set:08-17') ],
        [ kb_btn('10-19', 'ceo:settings:hours:set:10-19'), kb_btn('سفارشی (بعداً)', 'ceo:settings:hours') ],
        [ kb_btn('◀️ بازگشت','ceo:settings') ],
    ]);
    tg_send_kb_html($chat_id, "🕰 بازهٔ ساعات کاری را انتخاب کن:", $kb);
}
function ceo_settings_hours_set(int $chat_id, string $range): void {
    $cfg=ceo_settings_load(); $cfg['business_hours']=$range; ceo_settings_save($cfg);
    tg_send_html($chat_id, "✅ ساعات کاری روی <b>{$range}</b> تنظیم شد.");
    ceo_settings_menu($chat_id);
}
function ceo_settings_sla_toggle(int $chat_id): void {
    $cfg=ceo_settings_load(); $cfg['sla_alerts']=!($cfg['sla_alerts']??true); ceo_settings_save($cfg);
    tg_send_html($chat_id, "✅ وضعیت هشدار SLA: ".(($cfg['sla_alerts']??true)?'روشن 🟢':'خاموش 🔴'));
    ceo_settings_menu($chat_id);
}
function ceo_settings_export_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('👥 پرسنل', 'ceo:settings:export:employees'), kb_btn('🤝 مشتریان', 'ceo:settings:export:customers') ],
        [ kb_btn('🧾 فاکتورها', 'ceo:settings:export:invoices'), kb_btn('💼 دفترکل', 'ceo:settings:export:cashbook') ],
        [ kb_btn('📝 صورتجلسه‌ها', 'ceo:settings:export:minutes'), kb_btn('📦 همه باهم (ZIP)', 'ceo:settings:export:all') ],
        [ kb_btn('◀️ بازگشت','ceo:settings') ],
    ]);
    tg_send_kb_html($chat_id, "💾 چه چیزی خروجی بگیرم؟", $kb);
}
function ceo_settings_export_do(int $chat_id, string $what): void {
    if (!is_dir(EXPORT_DIR)) @mkdir(EXPORT_DIR, 0775, true);
    $ts = date('Ymd_His');
    $done = false; $path='';

    if ($what==='employees') {
        $path = EXPORT_DIR."/employees_{$ts}.json";
        $done = (bool)@file_put_contents($path, json_encode(storage_load(EMPLOYEES_FILE, []), JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
    } elseif ($what==='customers') {
        $path = EXPORT_DIR."/customers_{$ts}.json";
        $done = (bool)@file_put_contents($path, json_encode(storage_load(CUSTOMERS_FILE, []), JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
    } elseif ($what==='invoices') {
        $path = EXPORT_DIR."/invoices_{$ts}.json";
        $done = (bool)@file_put_contents($path, json_encode(inv_list(), JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
    } elseif ($what==='cashbook') {
        $file = __DIR__.'/../storage/cashbook.json';
        $path = EXPORT_DIR."/cashbook_{$ts}.json";
        $done = (bool)@file_put_contents($path, is_file($file)? (string)@file_get_contents($file) : '[]');
    } elseif ($what==='minutes') {
        $file = __DIR__.'/../storage/minutes.json';
        $path = EXPORT_DIR."/minutes_{$ts}.json";
        $done = (bool)@file_put_contents($path, is_file($file)? (string)@file_get_contents($file) : '[]');
    } elseif ($what==='all') {
        // ZIP از پوشه storage (اگر ZipArchive باشد)
        if (class_exists('ZipArchive')) {
            $zipPath = EXPORT_DIR."/backup_{$ts}.zip";
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE|ZipArchive::OVERWRITE) === true) {
                $root = realpath(__DIR__.'/../storage');
                $it = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS));
                foreach ($it as $file) {
                    $filePath = (string)$file;
                    $local = substr($filePath, strlen($root) + 1);
                    $zip->addFile($filePath, $local);
                }
                $zip->close();
                $done = true; $path = $zipPath;
            }
        } else {
            tg_send_html($chat_id, "⛔️ ZipArchive روی هاست فعال نیست. از گزینه‌های JSON استفاده کن.");
            return;
        }
    }

    if ($done) {
        tg_send_html($chat_id, "✅ خروجی آماده شد.\nمسیر فایل روی سرور:\n<code>".tg_escape($path)."</code>");
    } else {
        tg_send_html($chat_id, "⛔️ خروجی ساخته نشد. سطح دسترسی پوشه <code>storage/exports</code> را بررسی کن.");
    }
}

/* ────────── کال‌بک‌ها ────────── */
if (!function_exists('ceo_handle_callback')) {
    function ceo_handle_callback(int $chat_id, int $user_id, string $data): bool {
        if (!ceo_is_ceo($user_id) || !str_starts($data,'ceo:')) return false;

        /* سطح اول */
        if ($data==='ceo:back')       { ceo_menu_main($chat_id); return true; }
        if ($data==='ceo:dashboard')  { ceo_dashboard_render($chat_id); return true; }

        if ($data==='ceo:reports') {
            $kb = kb_inline([
                [ kb_btn('📩 روزانه', 'ceo:rep:daily'), kb_btn('📅 هفتگی', 'ceo:rep:weekly') ],
                [ kb_btn('📈 ماهانه', 'ceo:rep:monthly') ],
                [ kb_btn('◀️ بازگشت', 'ceo:back') ],
            ]);
            tg_send_kb_html($chat_id,"📝 <b>گزارش‌ها</b>", $kb); 
            return true;
        }
        if ($data==='ceo:rep:daily')   { ceo_report_quick($chat_id,'روزانه'); return true; }
        if ($data==='ceo:rep:weekly')  { ceo_report_quick($chat_id,'هفتگی'); return true; }
        if ($data==='ceo:rep:monthly') { ceo_report_quick($chat_id,'ماهانه'); return true; }

        if ($data==='ceo:staff') {
            $kb = kb_inline([
                [ kb_btn('📋 لیست پرسنل', 'ceo:staff:list') ],
                [ kb_btn('➕ افزودن پرسنل (راهنما)', 'ceo:staff:add') ],
                [ kb_btn('◀️ بازگشت', 'ceo:back') ],
            ]);
            tg_send_kb_html($chat_id,"👥 <b>مدیریت پرسنل</b>\nانتخاب کن:", $kb); 
            return true;
        }
        if ($data==='ceo:customers') {
            $kb = kb_inline([
                [ kb_btn('📋 لیست مشتریان', 'ceo:cust:list') ],
                [ kb_btn('➕ افزودن مشتری (راهنما)', 'ceo:cust:add') ],
                [ kb_btn('◀️ بازگشت', 'ceo:back') ],
            ]);
            tg_send_kb_html($chat_id,"🤝 <b>مدیریت مشتریان</b>", $kb); 
            return true;
        }
        if ($data==='ceo:departments') {
            $kb = kb_inline([
                [ kb_btn('👤 لیست مدیران دپارتمان', 'ceo:dept:managers') ],
                [ kb_btn('🔄 تخصیص مشتری به مدیر', 'ceo:dept:assign') ],
                [ kb_btn('◀️ بازگشت', 'ceo:back') ],
            ]);
            tg_send_kb_html($chat_id,"🏢 <b>مدیریت دپارتمان‌ها</b>", $kb); 
            return true;
        }

        /* 🧾 حسابداری: مدیرعامل → پنل کامل حسابداری */
        if ($data==='ceo:accounting') {
            if (function_exists('accountant_menu_main')) {
                accountant_menu_main($chat_id);
            } else {
                tg_send_html($chat_id,'⛔️ پنل حسابداری نصب/فعال نیست.');
            }
            return true;
        }

        /* پرسنل */
        if ($data==='ceo:staff:list')                       { ceo_staff_list($chat_id,1); return true; }
        if (preg_match('~^ceo:staff:list:p:(\d+)$~',$data,$m)) { ceo_staff_list($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:staff:show:(\d+)$~',$data,$m))   { ceo_staff_profile($chat_id,(int)$m[1]); return true; }
        if ($data==='ceo:staff:add') {
            $txt = "برای افزودن سریع پرسنل فعلاً از دستور متنی استفاده کن:\n"
                 . "<code>/setjob userId نقش</code>\n"
                 . "مثال:\n"
                 . "<code>/setjob 287098219 مدیردپارتمان</code>\n"
                 . "پس از ثبت، پرسنل با نقش جدید فعال می‌شود و پنل مخصوصش باز خواهد شد ✨";
            tg_send_kb_html($chat_id,$txt, ceo_back_kb('ceo:staff'));
            return true;
        }
        if (preg_match('~^ceo:staff:toggle:(\d+)$~',$data,$m)) { $ok=ceo_staff_toggle((int)$m[1]); tg_send_html($chat_id,$ok?'✅ وضعیت تغییر کرد.':'⛔️ نشد.'); ceo_staff_profile($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:staff:role:(\d+)$~',$data,$m)) {
            $uid=(int)$m[1];
            $kb = kb_inline([
                [ kb_btn('مدیردپارتمان', "ceo:staff:role:set:{$uid}:مدیردپارتمان"), kb_btn('ادمین', "ceo:staff:role:set:{$uid}:ادمین") ],
                [ kb_btn('سناریو نویس', "ceo:staff:role:set:{$uid}:سناریو نویس"), kb_btn('تصویربردار', "ceo:staff:role:set:{$uid}:تصویربردار") ],
                [ kb_btn('تدوینگر', "ceo:staff:role:set:{$uid}:تدوینگر"), kb_btn('حسابدار', "ceo:staff:role:set:{$uid}:حسابدار") ],
                [ kb_btn('◀️ بازگشت', "ceo:staff:show:{$uid}") ],
            ]);
            tg_send_kb_html($chat_id,'نقش جدید را انتخاب کن:', $kb); return true;
        }
        if (preg_match('~^ceo:staff:role:set:(\d+):(.+)$~u',$data,$m)) { $ok=ceo_staff_set_role((int)$m[1], $m[2]); tg_send_html($chat_id,$ok?'✅ نقش تغییر کرد.':'⛔️ نشد.'); ceo_staff_profile($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:staff:pen:(\d+)$~',$data,$m)) {
            $uid=(int)$m[1];
            $kb = kb_inline([
                [ kb_btn('۲۵۰,۰۰۰ عمومی', "ceo:staff:pen:set:{$uid}:250000"), kb_btn('۱۰۰,۰۰۰', "ceo:staff:pen:set:{$uid}:100000") ],
                [ kb_btn('۳۰۰,۰۰۰', "ceo:staff:pen:set:{$uid}:300000"), kb_btn('۵۰۰,۰۰۰', "ceo:staff:pen:set:{$uid}:500000") ],
                [ kb_btn('۱,۰۰۰,۰۰۰', "ceo:staff:pen:set:{$uid}:1000000") ],
                [ kb_btn('◀️ بازگشت', "ceo:staff:show:{$uid}") ],
            ]);
            tg_send_kb_html($chat_id, 'مبلغ جریمه را انتخاب کن:', $kb); return true;
        }
        if (preg_match('~^ceo:staff:pen:set:(\d+):(\d+)$~',$data,$m)) { ceo_staff_pen_quick($chat_id,(int)$m[1], (int)$m[2], 'جریمه دستی مدیرعامل'); ceo_staff_profile($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:staff:rew:(\d+)$~',$data,$m)) {
            $uid=(int)$m[1];
            $kb = kb_inline([
                [ kb_btn('۱۰۰,۰۰۰', "ceo:staff:rew:set:{$uid}:100000"), kb_btn('۳۰۰,۰۰۰', "ceo:staff:rew:set:{$uid}:300000") ],
                [ kb_btn('۵۰۰,۰۰۰', "ceo:staff:rew:set:{$uid}:500000"), kb_btn('۱,۰۰۰,۰۰۰', "ceo:staff:rew:set:{$uid}:1000000") ],
                [ kb_btn('◀️ بازگشت', "ceo:staff:show:{$uid}") ],
            ]);
            tg_send_kb_html($chat_id,'مبلغ پاداش را انتخاب کن:', $kb); return true;
        }
        if (preg_match('~^ceo:staff:rew:set:(\d+):(\d+)$~',$data,$m)) { $ok=ceo_staff_rew_quick((int)$m[1], (int)$m[2], 'پاداش مدیرعامل'); tg_send_html($chat_id,$ok?'✅ پاداش ثبت شد.':'⛔️ نشد.'); ceo_staff_profile($chat_id,(int)$m[1]); return true; }

        /* مشتریان */
        if ($data==='ceo:cust:list') { ceo_customers_list($chat_id,1); return true; }
        if (preg_match('~^ceo:cust:list:p:(\d+)$~',$data,$m)) { ceo_customers_list($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:cust:show:(\d+)$~',$data,$m)) { ceo_customer_profile($chat_id,(int)$m[1]); return true; }
        if ($data==='ceo:cust:add') {
            tg_send_kb_html($chat_id,"برای افزودن سریع مشتری، یا مشتری از /start خودش را معرفی کند، یا از ابزار دمو استفاده کن.", ceo_back_kb('ceo:customers')); 
            return true;
        }

        /* دپارتمان‌ها */
        if ($data==='ceo:dept:managers') {
            tg_send_kb_html($chat_id, "لیست مدیران دپارتمان از مسیر «تخصیص مشتری → انتخاب مدیر» قابل مشاهده و انتخاب است.", ceo_back_kb('ceo:departments'));
            return true;
        }
        if ($data==='ceo:dept:assign') {
            $kb = kb_inline([[ kb_btn('ابتدا مشتری را از «📋 لیست مشتریان» انتخاب کن', 'ceo:cust:list') ], [ kb_btn('◀️ بازگشت','ceo:departments') ]]);
            tg_send_kb_html($chat_id,'راهنما: مشتری را انتخاب کن و داخل پروفایلش «👑 انتخاب/تغییر مدیر دپارتمان» را بزن.', $kb);
            return true;
        }
        if (preg_match('~^ceo:dept:assign:start:(\d+)$~',$data,$m)) {
            $cust=(int)$m[1];
            tg_send_kb_html($chat_id,'یک مدیر دپارتمان انتخاب کن:', ceo_dept_pick_manager_kb($cust,1));
            return true;
        }
        if (preg_match('~^ceo:dept:assign:pick:(\d+):p:(\d+)$~',$data,$m)) {
            $cust=(int)$m[1]; $p=(int)$m[2];
            tg_send_kb_html($chat_id,'یک مدیر دپارتمان انتخاب کن:', ceo_dept_pick_manager_kb($cust,$p));
            return true;
        }
        if (preg_match('~^ceo:dept:assign:set:(\d+):(\d+)$~',$data,$m)) {
            $cust=(int)$m[1]; $mgr=(int)$m[2];
            $team = dept_get_team($cust);
            $ok = dept_assign_team($cust, ['manager'=>$mgr] + $team);
            tg_send_html($chat_id, $ok? "✅ مدیر دپارتمان ".to_persian_num($mgr)." برای مشتری ".to_persian_num($cust)." ثبت شد." : "⛔️ انجام نشد.");
            ceo_customer_profile($chat_id,$cust);
            return true;
        }

        /* حسابداری داخل CEO */
        if ($data==='ceo:acc:invoices') { ceo_acc_list_invoices($chat_id,1,null); return true; }
        if (preg_match('~^ceo:acc:inv:p:(\d+)$~',$data,$m)) { ceo_acc_list_invoices($chat_id,(int)$m[1],null); return true; }
        if (preg_match('~^ceo:acc:inv:for:(\d+)$~',$data,$m)) { ceo_acc_list_invoices($chat_id,1,(int)$m[1]); return true; }
        if (preg_match('~^ceo:acc:inv:view:(\d+)$~',$data,$m)) { ceo_acc_view_invoice($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^ceo:acc:inv:settle:(\d+)$~',$data,$m)) { ceo_acc_settle_quick($chat_id,(int)$m[1]); return true; }

        /* Broadcast / Audit */
        if ($data==='ceo:broadcast') { ceo_broadcast_menu($chat_id); return true; }
        if ($data==='ceo:audit')     { ceo_audit_show($chat_id,50); return true; }
        if ($data==='ceo:bc:aud:employees') { ceo_broadcast_start($chat_id,$user_id,'employees'); return true; }
        if ($data==='ceo:bc:aud:managers')  { ceo_broadcast_start($chat_id,$user_id,'managers');  return true; }
        if ($data==='ceo:bc:aud:customers') { ceo_broadcast_start($chat_id,$user_id,'customers'); return true; }

        /* Settings */
        if ($data==='ceo:settings')                { ceo_settings_menu($chat_id); return true; }
        if ($data==='ceo:settings:sla:toggle')     { ceo_settings_sla_toggle($chat_id); return true; }
        if ($data==='ceo:settings:hours')          { ceo_settings_hours_menu($chat_id); return true; }
        if (preg_match('~^ceo:settings:hours:set:(\d{2}-\d{2})$~',$data,$m)) { ceo_settings_hours_set($chat_id,$m[1]); return true; }
        if ($data==='ceo:settings:export')         { ceo_settings_export_menu($chat_id); return true; }
        if (preg_match('~^ceo:settings:export:(employees|customers|invoices|cashbook|minutes|all)$~',$data,$m)) { ceo_settings_export_do($chat_id,$m[1]); return true; }

        tg_send_html($chat_id,'دستور شناخته نشد.');
        return true;
    }
}

/* ────────── هندل متن مدیرعامل ────────── */
if (!function_exists('ceo_handle_text')) {
    function ceo_handle_text(int $chat_id, int $user_id, string $text): bool {
        if (!ceo_is_ceo($user_id)) return false;

        // مرحله متن Broadcast
        $st = ceo_state_get($user_id,'bc');
        if ($st && ($st['step'] ?? '')==='await_text') {
            if (mb_strlen(trim($text)) < 3) { tg_send_html($chat_id,'⛔️ متن خیلی کوتاهه. دوباره بفرست.'); return true; }
            ceo_broadcast_send($chat_id, $user_id, $text);
            return true;
        }

        // پیش‌فرض: منوی اصلی
        ceo_menu_main($chat_id);
        return true;
    }
}
