<?php
declare(strict_types=1);

/**
 * VisionMedia Bot — index.php (ULTRA STABLE)
 * - Dynamic loader for /modules and /handlers
 * - Safe requires (no fatals if a file is missing)
 * - Health & self-test
 * - Robust routing with try/catch
 * - Always return HTTP 200 to Telegram
 */

ini_set('display_errors', '0');
error_reporting(E_ALL);

// ── tiny logger
if (!function_exists('app_log')) {
    function app_log(string $msg): void {
        $f = __DIR__ . '/storage/logs/app.log';
        if (!is_dir(dirname($f))) { @mkdir(dirname($f), 0775, true); }
        @file_put_contents($f, '['.date('Y-m-d H:i:s')."] ".$msg."\n", FILE_APPEND);
    }
}

// ── helper: safe require (no fatal)
function safe_require(string $path, bool $required = true): void {
    if (is_file($path)) {
        try { require_once $path; app_log("[BOOT] loaded: ".str_replace(__DIR__.'/', '', $path)); }
        catch (Throwable $e) { app_log("[FATAL] include $path -> ".$e->getMessage()); }
    } else {
        app_log("[WARN] require miss: ".str_replace(__DIR__.'/', '', $path));
        if ($required) {
            // required=false برای فایل‌های غیرحیاتی؛ این یکی رو هم fatal نکن، فقط لاگ کنیم
        }
    }
}

// ── base includes (order matters)
$BASE = __DIR__;
safe_require($BASE . '/config/env.php');
safe_require($BASE . '/core/utils.php');
safe_require($BASE . '/core/storage.php');
safe_require($BASE . '/core/telegram.php');

// ── dynamic load: modules/*.php
$mods = glob($BASE.'/modules/*.php') ?: [];
sort($mods);
foreach ($mods as $m) safe_require($m, false);

// ── dynamic load: handlers/*.php
$hands = glob($BASE.'/handlers/*.php') ?: [];
sort($hands);
foreach ($hands as $h) safe_require($h, false);

// ── polyfills
if (!function_exists('str_starts')) {
    function str_starts(string $haystack, string $needle): bool {
        return strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

// ── Health & Self-Test (open in browser)
if ($_SERVER['REQUEST_METHOD'] === 'GET' && empty($_POST)) {
    if (isset($_GET['test'])) {
        $t = strtolower((string)$_GET['test']);
        if ($t === 'hello') {
            if (defined('CEO_UID')) {
                @tg_send_html((int)CEO_UID, "✅ Self-Test: Bot alive at ".date('Y-m-d H:i:s'));
            }
            header('Content-Type: text/plain; charset=utf-8');
            echo 'SENT '.date('Y-m-d H:i:s'); http_response_code(200); exit;
        }
        if ($t === 'ping') {
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode([
                'ok'=>true,
                'time'=>date('Y-m-d H:i:s'),
                'php'=>PHP_VERSION,
                'webhook_base'=>defined('WEBHOOK_BASE')? WEBHOOK_BASE : null
            ], JSON_UNESCAPED_UNICODE);
            http_response_code(200); exit;
        }
    }
    header('Content-Type: text/plain; charset=utf-8');
    echo 'OK '.date('Y-m-d H:i:s'); http_response_code(200); exit;
}

// ── read update
$raw = file_get_contents('php://input');
if ($raw === '' || $raw === false) {
    app_log('[BOOT] empty body'); http_response_code(200); exit;
}
$update = json_decode($raw, true);
if (!is_array($update)) {
    app_log('[BOOT] invalid json'); http_response_code(200); exit;
}
try {
    $kind = implode(',', array_keys($update));
} catch (Throwable $e) { $kind = 'unknown'; }
app_log('[UPD] kind='.$kind.(isset($update['callback_query']['data'])? ' data='.$update['callback_query']['data'] : ''));

// ignore membership updates
if (isset($update['my_chat_member']) || isset($update['chat_member'])) {
    app_log('[UPD] member change'); http_response_code(200); exit;
}

/* ==== ROUTING ==== */

// helper: call a function if exists, catch errors, return bool
function try_call(string $fn, array $args): bool {
    if (!function_exists($fn)) return false;
    try {
        $res = call_user_func_array($fn, $args);
        return (bool)$res;
    } catch (Throwable $e) {
        app_log('[ERR] '.$fn.' -> '.$e->getMessage().' @ '.$e->getFile().':'.$e->getLine());
        return false;
    }
}

// ── CALLBACK QUERY
if (isset($update['callback_query'])) {
    $cq      = $update['callback_query'];
    $data    = (string)($cq['data'] ?? '');
    $chat_id = (int)($cq['message']['chat']['id'] ?? 0);
    $user_id = (int)($cq['from']['id'] ?? 0);

    $handled = false;

    // اول CEO، بعد Accounting، بعد سایرین؛ هر کدوم نبود، مشکلی نیست
    $handled = $handled || try_call('ceo_handle_callback',            [$chat_id, $user_id, $data]);
    $handled = $handled || try_call('accountant_handle_callback',     [$chat_id, $user_id, $data]);
    $handled = $handled || try_call('accountant_handle_callback_more',[$chat_id, $user_id, $data]);
    // در صورت نیاز: $handled = $handled || try_call('department_handle_callback', ...);

    // همیشه پاسخ بده تا spinner نایستد
    try {
        if (!$handled) tg_answer_cb('دستور نامعتبر یا منقضی شده.', true);
        else tg_answer_cb('');
    } catch (Throwable $e) {
        app_log('[ERR] answerCb -> '.$e->getMessage());
    }

    http_response_code(200); exit;
}

// ── MESSAGE / TEXT
if (isset($update['message']) && isset($update['message']['text'])) {
    $msg     = $update['message'];
    $text    = trim((string)$msg['text']);
    $chat_id = (int)($msg['chat']['id'] ?? 0);
    $user_id = (int)($msg['from']['id'] ?? 0);

    $handled = false;

    // start → ceo → accounting → سایرین
    $handled = $handled || try_call('start_handle_text',        [$chat_id, $user_id, $text]);
    $handled = $handled || try_call('ceo_handle_text',          [$chat_id, $user_id, $text]);
    $handled = $handled || try_call('accountant_handle_text',   [$chat_id, $user_id, $text]);
    // $handled = $handled || try_call('department_handle_text', [$chat_id, $user_id, $text]);

    if (!$handled) {
        // fallback مهربان بدون قطع امکانات قبلی
        if (defined('CEO_UID') && $user_id === (int)CEO_UID && function_exists('ceo_menu_main')) {
            try_call('ceo_menu_main', [$chat_id]);
        } elseif (function_exists('accountant_menu_main') && function_exists('acc_is_accountant_or_ceo') && acc_is_accountant_or_ceo($user_id)) {
            try_call('accountant_menu_main', [$chat_id]);
        } elseif (function_exists('start_menu')) {
            try_call('start_menu', [$chat_id]);
        } else {
            try { tg_send_html($chat_id, "برای شروع از /start استفاده کن ✨"); } catch (Throwable $e) { app_log('[ERR] send fallback -> '.$e->getMessage()); }
        }
    }

    http_response_code(200); exit;
}

// ── ANY OTHER UPDATE → 200
http_response_code(200);
exit;
